#ifdef _WIN32
#include <winsock2.h>
#include <windows.h>
#else
#define _MULTI_THREADED
#include <unistd.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <pthread.h>
#include <semaphore.h>
typedef int SOCKET;
#endif
#include "lua.h"
#include "lauxlib.h"
#include <string.h>
#include <stdlib.h>

#define THREAD_THREADDATA           "tThreadData**"
#define THREAD_COROUTINE_TO_NATIVE  "thread_LuaState2Native"
#define THREAD_IDLE_NATIVE          "thread_IdleNative"
#define THREAD_COMMON_DATA          "thread_CommonData"

/* Small emulation of pthread.h and semaphore.h for Windows */
#ifdef _WIN32
typedef HANDLE sem_t;
typedef HANDLE pthread_t;
static int sem_init(sem_t *sem, int pshared, unsigned int value)
{ 
  *sem = CreateSemaphore(NULL, value, 10000000, NULL); 
  return (*sem == NULL ? -1 : 0);
}
static void sem_post(sem_t* sem) { ReleaseSemaphore(*sem, 1, NULL); }
static void sem_wait(sem_t* sem) { WaitForSingleObject(*sem, INFINITE); }
static void sem_destroy(sem_t* sem) { CloseHandle(*sem); }
static int pthread_create(pthread_t *thread, void* dummy, LPTHREAD_START_ROUTINE fct, void *arg)
{
  DWORD id;
  *thread = CreateThread(NULL, 2000, fct, arg, 0, &id);
  return (*thread == NULL ? -1 : 0);
}
static int pthread_cancel(pthread_t thread)
{
  TerminateThread(thread, 1);
  CloseHandle(thread);
  return 0;
}
#endif

/* Definition of data types and functions */
typedef enum { ERROR_LUA_STATE, ERROR_STRING } eErrorMode;
typedef enum { THREAD_IDLE, THREAD_REQUEST, THREAD_WORKING, 
  THREAD_FINISHED, THREAD_NB_STATES } eThreadState;

typedef struct tThreadMessage
{
  int (*Fct)(struct tThreadMessage* msg);
  lua_State* ErrorL;
  const char* ErrorString;
  int IntegerValue;
  const char* InputString;
  const char* InputString2;
  int OutputSize;
  char* OutputString;
  SOCKET Socket;
} tThreadMessage;

typedef struct tThreadData
{
  pthread_t Thread;
  sem_t StartSemaphore;
  sem_t FinishedSemaphore;
  tThreadMessage Message;
  eThreadState State;
  lua_State* LuaCoroutine;
  struct tCommonData* Common;
  struct tThreadData* pNext;
  struct tThreadData* pPrevious;
} tThreadData;

typedef struct tList
{
  tThreadData* pFirst;
  int Count;
} tList;

typedef struct tCommonData
{
  tList List[THREAD_NB_STATES];
  sem_t Mutex;
  sem_t EndSemaphore;
} tCommonData;

static int exchange_with_thread(lua_State* L, tThreadMessage* msg);
static tThreadData* retrieve_thread_data(lua_State* L);

/* User function implementation */
static void close_socket(tThreadMessage* msg)
{
  if(msg->Socket)
  {
#ifdef _WIN32
    closesocket(msg->Socket);
#else
    close(msg->Socket);
#endif
    msg->Socket = 0;
  }
}

static void close_buffer(tThreadMessage* msg)
{
  free(msg->OutputString);
  msg->OutputString = NULL;
}

static int error(tThreadMessage* msg, const char* text)
{
  close_buffer(msg);
  close_socket(msg);
  if(msg->ErrorL)
    luaL_error(msg->ErrorL, "%s", text);
  else
    msg->ErrorString = text;
  return 0;
}

static int execute_sleep(tThreadMessage* msg)
{
#ifdef _WIN32
  Sleep(msg->IntegerValue);
#else
  usleep(msg->IntegerValue * 1000);
#endif
  return 1;
}

static int thread_sleep(lua_State* L)
{
  tThreadMessage msg;
  memset(&msg, 0, sizeof(msg));
  msg.Fct = execute_sleep;
  msg.IntegerValue = luaL_checkinteger(L, 1);
  return exchange_with_thread(L, &msg);
}

static int execute_gets(tThreadMessage* msg)
{
  msg->OutputString = (char*)malloc(msg->OutputSize);
  if(fgets(msg->OutputString, msg->OutputSize, stdin) == NULL)
    return error(msg, "Error reading line from sdtin");
  return 1;
}

static int thread_gets(lua_State* L)
{
  tThreadMessage msg;
  memset(&msg, 0, sizeof(msg));
  msg.Fct = execute_gets;
  msg.OutputSize = luaL_optinteger(L, 1, 1000);
  if(exchange_with_thread(L, &msg) < 0)
    return -1;
  lua_pushstring(L, msg.OutputString);
  close_buffer(&msg);
  return 1;
}

static int execute_system(tThreadMessage* msg)
{
  msg->IntegerValue = system(msg->InputString);
  if(msg->IntegerValue < 0)
    return error(msg, "Error running command");
  return 1;
}

static int thread_system(lua_State* L)
{
  tThreadMessage msg;
  memset(&msg, 0, sizeof(msg));
  msg.Fct = execute_system;
  msg.InputString = luaL_optstring(L, 1, NULL);
  if(exchange_with_thread(L, &msg) < 0)
    return -1;
  lua_pushinteger(L, msg.IntegerValue);
  return 1;
}

static int execute_wget(tThreadMessage* msg)
{
  struct sockaddr_in sin;
  struct hostent *h;
  const char* prequest;
  int buffer_size = 10000;
  if ((msg->Socket = socket(AF_INET, SOCK_STREAM,0)) < 0) 
    return error(msg, "Error opening socket");
  memset (& sin, 0, sizeof (sin));
  sin.sin_family = AF_INET;
  h = gethostbyname(msg->InputString);
  if(h == NULL)
    return error(msg, "Cannot find host name");
  sin.sin_addr = *((struct in_addr *) h->h_addr_list[0]);
  sin.sin_port = htons((unsigned short)msg->IntegerValue);
  if (connect(msg->Socket, (struct sockaddr *) & sin, sizeof (sin)) < 0) 
    return error(msg, "Error connecting socket");
  msg->OutputString = (char*)malloc(buffer_size);
  if(strchr(msg->InputString2, '\n'))
    prequest = msg->InputString2;
  else
  {
    sprintf(msg->OutputString, "GET %s HTTP/1.0\r\n\r\n", msg->InputString2);
    prequest = msg->OutputString;
  }
  if(send(msg->Socket, prequest, strlen(prequest), 0) < 0)
    return error(msg, "Cannot send request to socket");
  while(1)
  {
    int ret;
    if(msg->OutputSize >= buffer_size)
    {
      buffer_size *= 2;
      msg->OutputString = (char*)realloc(msg->OutputString, buffer_size);
    }
    ret = recv(msg->Socket, msg->OutputString + msg->OutputSize, 
      buffer_size - msg->OutputSize, 0);
    if(ret > 0)
      msg->OutputSize += ret;
    else if(ret == 0)
      break;
    else
      return error(msg, "Cannot receive data from socket");
  }
  close_socket(msg);
  return 1;
}

static int thread_wget(lua_State* L)
{
  tThreadMessage msg;
  memset(&msg, 0, sizeof(msg));
  msg.Fct = execute_wget;
  msg.InputString = luaL_checkstring(L, 1);
  msg.InputString2 = luaL_checkstring(L, 2);
  msg.IntegerValue = luaL_optinteger(L, 3, 80);
  if(exchange_with_thread(L, &msg) < 0)
    return -1;
  lua_pushlstring(L, msg.OutputString, msg.OutputSize);
  close_buffer(&msg);
  return 1;
}

/* Functions to manage a double linked list */
static void insert_in_list(tList* list, tThreadData* elem)
{
  list->Count++;
  if(list->pFirst == NULL)
  {
    list->pFirst = elem;
    elem->pPrevious = elem->pNext = elem;
  }
  else
  {
    tThreadData* top = list->pFirst;
    elem->pNext = top->pNext;
    elem->pNext->pPrevious = elem;
    top->pNext = elem;
    elem->pPrevious = top;
  }
}
static void remove_from_list(tList* list, tThreadData* elem)
{
  list->Count--;
  elem->pPrevious->pNext = elem->pNext;
  elem->pNext->pPrevious = elem->pPrevious;
  if(elem == list->pFirst)
  {
    if(elem == elem->pNext)
      list->pFirst = NULL;
    else
      list->pFirst = elem->pNext;
  }
  elem->pNext = elem->pPrevious = NULL;
}

/* Core multithreading functions */
static void set_thread_state(tThreadData* pthread, eThreadState state)
{
  tCommonData* common = pthread->Common;
  if(state == THREAD_REQUEST)
    sem_post(&pthread->StartSemaphore);
  else if(state == THREAD_IDLE)
    sem_wait(&pthread->FinishedSemaphore);
  sem_wait(&common->Mutex);
  remove_from_list(&common->List[pthread->State], pthread);
  insert_in_list(&common->List[state], pthread);
  pthread->State = state;
  sem_post(&common->Mutex);
}

static void register_idle_thread(lua_State* L, tThreadData* pthread, int idle)
{
  lua_getfield(L, LUA_REGISTRYINDEX, THREAD_IDLE_NATIVE);
  lua_pushlightuserdata(L, pthread);
  if(!idle)
  {
    *(tThreadData**)lua_newuserdata(L, sizeof(tThreadData*)) = pthread;
    luaL_getmetatable(L, THREAD_THREADDATA);
    lua_setmetatable(L, -2);
  }
  else
    lua_pushboolean(L, 1);
  lua_settable(L, -3);
  lua_pop(L, 1);
}

static int exchange_with_thread(lua_State* L, tThreadMessage* msg)
{
  tThreadData* pthread = retrieve_thread_data(L);
  if(pthread == NULL)
  {
    msg->ErrorL = L;
    msg->Fct(msg);
    return 0;
  }
  switch(pthread->State)
  {
  case THREAD_IDLE:
    pthread->Message = *msg;
    set_thread_state(pthread, THREAD_REQUEST);
    register_idle_thread(L, pthread, 0);
    return lua_yield(L, 0);
  case THREAD_FINISHED:
    *msg = pthread->Message;
    set_thread_state(pthread, THREAD_IDLE);
    register_idle_thread(L, pthread, 1);
    if(msg->ErrorString)
      luaL_error(L, "%s", msg->ErrorString);
    return 0;
  default:
    return lua_yield(L, 0);
  }
}

#ifdef _WIN32
static DWORD WINAPI thread_main(LPVOID lpParameter)
#else
static void* thread_main(void* lpParameter)
#endif
{
  tThreadData* This = (tThreadData*)lpParameter;
  while(1)
  {
    sem_wait(&This->StartSemaphore);
    set_thread_state(This, THREAD_WORKING);
    if(This->Message.Fct)
      This->Message.Fct(&This->Message);
    set_thread_state(This, THREAD_FINISHED);
    sem_post(&This->FinishedSemaphore);
    sem_post(&This->Common->EndSemaphore);
  }
}

static tThreadData* retrieve_thread_data(lua_State* L)
{
  tThreadData* result;
  tCommonData* common;
  lua_getfield(L, LUA_REGISTRYINDEX, THREAD_COMMON_DATA);
  common = (tCommonData*)lua_touserdata(L, -1);
  lua_getfield(L, LUA_REGISTRYINDEX, THREAD_COROUTINE_TO_NATIVE);
  if(lua_pushthread(L))
    return NULL; /* Main thread */
  lua_gettable(L, -2);
  result = (tThreadData*)lua_topointer(L, -1);
  if(result == NULL)
  {
    sem_wait(&common->Mutex);
    result = common->List[THREAD_IDLE].pFirst;
    sem_post(&common->Mutex);
    if(result == NULL)
    {
      result = (tThreadData*)malloc(sizeof(tThreadData)); 
      memset(result, 0, sizeof(tThreadData));
      sem_init(&result->StartSemaphore, 0, 0);
      sem_init(&result->FinishedSemaphore, 0, 0);
      if(pthread_create(&result->Thread, NULL, thread_main, result) != 0)
        luaL_error(L, "unable to create new thread");
      result->Common = common;
      sem_wait(&common->Mutex);
      insert_in_list(&common->List[THREAD_IDLE], result);
      sem_post(&common->Mutex);
    }
    lua_pushthread(L);
    lua_pushlightuserdata(L, result);
    lua_settable(L, -4);
  }
  if(result->State == THREAD_FINISHED)
  {
    lua_pushthread(L);
    lua_pushnil(L);
    lua_settable(L, -4);
  }
  result->LuaCoroutine = L;
  lua_pop(L, 3);
  return result;
}

static int thread_wait(lua_State* L)
{
  tThreadData* pthread;
  tCommonData* common;
  int nbthreads, i;
  lua_getfield(L, LUA_REGISTRYINDEX, THREAD_COMMON_DATA);
  common = (tCommonData*)lua_touserdata(L, -1);
  while(1)
  {
    sem_wait(&common->Mutex);
    pthread = common->List[THREAD_FINISHED].pFirst;
    for(nbthreads=0,i=THREAD_REQUEST;i<=THREAD_FINISHED;i++)
      nbthreads += common->List[i].Count;
    sem_post(&common->Mutex);
    if(nbthreads == 0)
      return 0;
    if(pthread)
    {
      lua_pushthread(pthread->LuaCoroutine);
      lua_xmove(pthread->LuaCoroutine, L, 1);
      return 1;
    }
    sem_wait(&common->EndSemaphore);
  }
}

static int thread_gc(lua_State* L)
{
  int type;
  tThreadData* pthread = *(tThreadData**)lua_topointer(L, 1);
  tCommonData* common = pthread->Common;
  lua_getfield(L, LUA_REGISTRYINDEX, THREAD_IDLE_NATIVE);
  lua_pushlightuserdata(L, pthread);
  lua_gettable(L, -2);
  type = lua_type(L, -1);
  if(type == LUA_TUSERDATA || type == LUA_TNIL)
    return 0; /* Still in use or already freed */
  lua_pushlightuserdata(L, pthread);
  lua_pushnil(L);
  lua_settable(L, -4);
  sem_wait(&common->Mutex);
  remove_from_list(&common->List[THREAD_IDLE], pthread);
  sem_post(&common->Mutex);
  pthread_cancel(pthread->Thread);
  sem_destroy(&pthread->StartSemaphore);
  sem_destroy(&pthread->FinishedSemaphore);
  free(pthread);
  return 0;
}

/* Initialization */
static const luaL_Reg thread_lib[] = {
  {"wait",     thread_wait},
  {NULL, NULL}
};

static const luaL_Reg thread_user_lib[] = {
  {"sleep",    thread_sleep},
  {"gets",     thread_gets},
  {"system",   thread_system},
  {"wget",     thread_wget},
  {NULL, NULL}
};

static const char init_thread_lua[] = 
  "for name,fct in pairs(thread) do\n"
  " thread[name] = function (...)\n"
  "   local result\n"
  "   repeat result = { fct(...) }\n"
  "   until result[1] ~= thread\n"
  "   return unpack(result)\n"
  "end end\n";

LUALIB_API int luaopen_thread (lua_State *L) {
  tCommonData* common;
#ifdef _WIN32
  WSADATA data;
  WSAStartup(MAKEWORD(2,0),&data);
#endif
  luaL_loadbuffer(L, init_thread_lua, strlen(init_thread_lua), "@thread");
  luaL_register(L, "thread", thread_user_lib);
  lua_pcall(L, 1, 0, 0);
  luaL_register(L, "thread", thread_lib);
  lua_createtable(L, 0, 0);
  lua_createtable(L, 0, 1);
  lua_setfield(L, LUA_REGISTRYINDEX, THREAD_COROUTINE_TO_NATIVE);
  lua_createtable(L, 0, 1);
  lua_setfield(L, LUA_REGISTRYINDEX, THREAD_IDLE_NATIVE);
  luaL_newmetatable(L, THREAD_THREADDATA);
  lua_pushcfunction(L, thread_gc);
  lua_setfield(L, -2, "__gc");
  common = (tCommonData*)lua_newuserdata(L, sizeof(tCommonData));
  memset(common, 0, sizeof(tCommonData));
  sem_init(&common->Mutex, 0, 1);
  sem_init(&common->EndSemaphore, 0, 0);
  lua_setfield(L, LUA_REGISTRYINDEX, THREAD_COMMON_DATA);
  return 1;
}
